const { Sequelize } = require("../config/db.connection");
const ContactUsModel = require("../model/contactUsModel");
const newsLetterModel = require("../model/newsLetterModel");
const {
    subscribedClientMail,
    subscribedAdminMail,
    contactAdminMail,
    contactClientMail,
    unsubscriptionAdminMail,
    clientNewsletterMail,
} = require("../utils/notificationMail");

const contactUsApi = async (req, res) => {
    const {
        name,
        email,
        mobile,
        comment,
        product,
        browserName,
        osName,
        ipAddress,
        countryName,
        type,
        recapcha,
    } = req.body;

    try {

        const isUnique = await ContactUsModel.checkRecapchaUnique(recapcha);

        if (!isUnique) {
            return res.status(400).json({ message: 'Recapcha must be unique.' });
        }

        const newContact = await ContactUsModel.create({
            name,
            email,
            mobile,
            comment,
            product,
            browserName,
            osName,
            ipAddress,
            countryName,
            type,
            recapcha,
        });

        await Promise.all([
            contactAdminMail(req.body),
            // contactClientMail(req.body),
        ]);

        return res.status(200).json({
            message: "Contact submitted successfully!",
            contactData: {
                name: newContact.name,
                email: newContact.email,
                mobile: newContact.mobile,
                comment: newContact.comment,
                product: newContact.product,
                browserName: newContact.browserName,
                osName: newContact.osName,
                ipAddress: newContact.ipAddress,
                countryName: newContact.countryName,
                type: newContact.type,
                recapcha: newContact.recapcha,
            },
        });
    } catch (error) {
        console.error("Error occurred:", error);
        return res.status(500).json({
            message: "Failed to submit contact. Please try again later.",
        });
    }
};

const newsLetterApi = async (req, res) => {
    const { email, browserName, osName, ipAddress, countryName, type } =
        req.body;

    try {
        const existingNewsLetter = await newsLetterModel.findOne({
            where: {
                email,
                isDeleted: "0",
            },
        });

        if (existingNewsLetter) {
            return res
                .status(409)
                .json({ message: "Email is already subscribed." });
        }

        const deletedNewsLetter = await newsLetterModel.findOne({
            where: {
                email,
                isDeleted: "1",
            },
        });

        if (deletedNewsLetter) {
            await deletedNewsLetter.update({ isDeleted: "0" });

            await Promise.all([
                subscribedClientMail(req.body),
                subscribedAdminMail(req.body),
            ]);

            return res.status(200).json({
                message: "Newsletter subscription reactivated successfully!",
                newsLetterData: {
                    email: deletedNewsLetter.email,
                },
            });
        }

        const newNewsLetter = await newsLetterModel.create({
            email,
            browserName,
            osName,
            ipAddress,
            countryName,
            type,
        });

        if (newNewsLetter) {
            await Promise.all([
                subscribedClientMail(req.body),
                subscribedAdminMail(req.body),
            ]);

            return res.status(200).json({
                message: "Newsletter subscription successful!",
                newsLetterData: {
                    email: newNewsLetter.email,
                    browserName: newNewsLetter.browserName,
                    osName: newNewsLetter.osName,
                    ipAddress: newNewsLetter.ipAddress,
                    countryName: newNewsLetter.countryName,
                    type: newNewsLetter.type,
                },
            });
        } else {
            return res.status(500).json({
                message: "Failed to create newsletter subscription.",
            });
        }
    } catch (error) {
        console.error("Error creating newsletter:", error);
        return res
            .status(500)
            .json({ message: "Failed to submit newsletter." });
    }
};

const clientNewsletterApi = async (req, res) => {
    const { email } = req.body;

    try {
        const existingNewsletter = await newsLetterModel.findOne({
            where: {
                email,
                isDeleted: { [Sequelize.Op.in]: ["0", "1"] },
            },
        });

        if (existingNewsletter) {
            if (existingNewsletter.isDeleted === "0") {
                return res
                    .status(409)
                    .json({ message: "Email is already subscribed." });
            } else {
                await clientNewsletterMail(req.body);
                return res.json({ isSubscribed: false });
            }
        }

        await clientNewsletterMail(req.body);
        return res.status(200).json({
            message:
                "Newsletter subscription confirmation mail sent successfully!",
            newsLetterData: { email },
        });
    } catch (error) {
        console.error("Error processing newsletter subscription:", error);
        return res
            .status(500)
            .json({ message: "Failed to submit newsletter." });
    }
};

const unsubscribedFunction = async (req, res) => {
    const email = req.body.email;

    try {
        await unsubscriptionAdminMail(email);

        await newsLetterModel.update(
            { isDeleted: "1" },
            { where: { email, isDeleted: "0" } }
        );

        return res.status(200).send("You have successfully unsubscribed.");
    } catch (error) {
        console.error("Error during unsubscription:", error);
        return res.status(500).send("Failed to unsubscribe.");
    }
};

const checkSubscription = async (req, res) => {
    const email = req.body.email;

    try {
        const existingSubscription = await newsLetterModel.findOne({
            where: { email, isDeleted: "0" },
        });

        if (!existingSubscription) {
            return res.json({ isSubscribed: false });
        }

        return res.json({ isSubscribed: true });
    } catch (error) {
        console.error("Error checking subscription:", error);
        return res.status(500).send("Failed to check subscription.");
    }
};

module.exports = {
    contactUsApi,
    newsLetterApi,
    unsubscribedFunction,
    checkSubscription,
    clientNewsletterApi
};
