const Joi = require("joi");
const leoProfanity = require("leo-profanity");

leoProfanity.loadDictionary();

const contactUsSchema = Joi.object({
    name: Joi.string()
        .pattern(/^[a-zA-Z\s_'`–—.-]+$/, 'string.pattern.base')
        .required()
        .custom((value, helpers) => {
            if (leoProfanity.check(value)) {
                return helpers.error('string.profane', { value });
            }
            return value;
        })
        .custom((value, helpers) => {
            const hasHTMLTags = /<[^>]*>/g.test(value);
            if (hasHTMLTags) {
                return helpers.error('string.html', { value });
            }
            return value;
        })
        .pattern(/^[^!@#$%^&*()<>{}[\]\/+;]+$/, 'string.pattern.base')
        .messages({
            'string.empty': 'Please enter your name.',
            'string.pattern.base': 'Name can only contain English letters, underscores, spaces, dashes, apostrophes, and dots.',
            'string.profane': 'Name contains inappropriate language.',
            'string.html': 'Name cannot contain HTML or script tags.',
        }),

    email: Joi.string()
        .email({ tlds: { allow: false } })
        .pattern(/^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$/)
        .required()
        .messages({
            "string.empty": "Please enter email",
            "string.email": "Please enter a valid email address",
            "string.pattern.base": "Please enter a valid email address",
        }),

    mobile: Joi.string()
        .required()
        .pattern(/^[0-9+\-\(\)\s]+$/)
        .custom((value, helpers) => {
            const digitsOnly = value.replace(/\D/g, "");
            if (digitsOnly.length < 10) {
                return helpers.message(
                    "Mobile number must be at least 10 digits."
                );
            }
            if (digitsOnly.length > 20) {
                return helpers.message(
                    "Mobile number must not exceed 20 digits."
                );
            }
            if (/^0+$/.test(digitsOnly)) {
                return helpers.message("Mobile number cannot consist of all zeroes.");
            }
            const countryCodeLength = value.startsWith('+') ? value.split(/[\s\(\)\-]/)[0].length : 0;
            const numberPart = digitsOnly.slice(countryCodeLength);

            if (numberPart !== '' && numberPart === '0'.repeat(numberPart.length)) {
                return helpers.message("Mobile number cannot be all zeros after the country code.");
            }
            return value;
        })
        .messages({
            "string.empty": "Please enter your mobile number.",
            "string.pattern.base": "Please enter a valid mobile number.",
        }),

    comment: Joi.string()
        .optional()
        .allow("")
        .custom((value, helpers) => {
            if (leoProfanity.check(value)) {
                return helpers.error("string.profane", { value });
            }
            if (!/^[A-Za-z0-9\s.,!?$&()":%'*#@]+$/.test(value)) {
                return helpers.error("string.pattern.base", { value });
            }
            const hasURL = /(https?:\/\/[^\s]+)/.test(value);
            if (hasURL) {
                return helpers.error("string.uri", { value });
            }
            const hasHTMLTags = /<\/?[^>]+(>|$)/.test(value);
            if (hasHTMLTags) {
                return helpers.error("string.html", { value });
            }
            return value;
        })
        .messages({
            "string.base": "Comment must be a string",
            "string.profane": "Comment contains inappropriate language",
            "string.uri": "URLs are not allowed in the comment name.",
            "string.html": "Comment cannot contain HTML or script tags.",
            "string.pattern.base": "Comment can only contain English characters."
        }),

    product: Joi.string()
        .required()
        .custom((value, helpers) => {
            if (leoProfanity.check(value)) {
                return helpers.error("string.profane", { value });
            }
            const nonEnglishPattern = /[^\x00-\x7F]+/;
            if (nonEnglishPattern.test(value)) {
                return helpers.error("string.pattern.base", { value });
            }
            const hasURL = /(https?:\/\/[^\s]+)/.test(value);
            if (hasURL) {
                return helpers.error("string.uri", { value });
            }
            const hasHTMLTags = /<\/?[^>]+(>|$)/.test(value);
            if (hasHTMLTags) {
                return helpers.error("string.html", { value });
            }
            return value;
        })
        .messages({
            "string.base": "Product must be a string.",
            "string.profane": "Product name contains inappropriate language.",
            "any.required": "Product is required.",
            "string.uri": "URLs are not allowed in the product name.",
            "string.html": "Product cannot contain HTML or script tags.",
            "string.pattern.base": "Product can only contain English characters."
        }),

    browserName: Joi.string().optional().allow(""),

    osName: Joi.string().optional().allow(""),

    ipAddress: Joi.string().optional().allow(""),

    countryName: Joi.string().optional().allow(""),

    type: Joi.string().optional().allow(""),

    recapcha: Joi.string().required(),
});

module.exports = { contactUsSchema };
